// cases.js loaded globally

document.addEventListener('alpine:init', () => {
    Alpine.data('simulator', () => ({
        // State
        view: 'welcome', // welcome, sector-select, cases-list, simulation, results
        playerName: '',
        selectedSector: '',
        availableCases: [],
        currentCase: null,
        currentStepId: 0,
        currentStep: null,
        tempScores: {},
        userStats: {
            'Comunicación': 0,
            'Liderazgo': 0,
            'Trabajo en Equipo': 0,
            'Toma de Decisiones': 0
        },
        feedbackMsg: '',
        feedbackType: '', // good, bad, neutral
        showFeedbackModal: false,
        showAbandonModal: false,
        showNameErrorModal: false,
        sessionCompleted: false,

        init() {
            // Check local storage for previous session? (Optional)
        },

        startSession() {
            if (this.playerName.trim() === '') {
                this.showNameErrorModal = true;
                return;
            }
            this.view = 'sector-select';
        },

        abandonSession() {
            this.showAbandonModal = true;
        },

        confirmAbandon() {
            this.view = 'welcome';
            this.playerName = '';
            this.selectedSector = '';
            this.availableCases = [];
            this.userStats = {
                'Comunicación': 0,
                'Liderazgo': 0,
                'Trabajo en Equipo': 0,
                'Toma de Decisiones': 0
            };
            this.showAbandonModal = false;
        },

        cancelAbandon() {
            this.showAbandonModal = false;
        },

        selectSector(sector) {
            this.selectedSector = sector;
            this.loadCases();
            this.view = 'cases-list';
        },

        loadCases() {
            // Filter by sector or Transversal
            const cases = window.ScenarioBank || [];
            let filtered = cases.filter(c => c.area === this.selectedSector || c.area === 'Transversal');

            // If not enough specific cases, fill with others randomized
            if (filtered.length < 5) {
                const others = cases.filter(c => c.area !== this.selectedSector && c.area !== 'Transversal');
                const needed = 5 - filtered.length;
                filtered = [...filtered, ...others.sort(() => 0.5 - Math.random()).slice(0, needed)];
            } else {
                // Shuffle and pick 5
                filtered = filtered.sort(() => 0.5 - Math.random()).slice(0, 5);
            }

            this.availableCases = filtered.map(c => ({ ...c, completed: false })); // Deep copy enough for this
        },

        selectCase(caseId) {
            this.currentCase = this.availableCases.find(c => c.id === caseId);
            this.currentStepId = 0;
            this.updateStep();
            this.tempScores = {};
            this.view = 'simulation';
        },

        updateStep() {
            this.currentStep = this.currentCase.story.find(s => s.id === this.currentStepId);
            if (this.currentStep.options.length === 0) {
                // End of scenario reached
                this.finishCase(true, this.currentStep.feedbackText);
            }
        },

        chooseOption(option) {
            // Calculate scores
            for (const [skill, value] of Object.entries(option.scores)) {
                this.tempScores[skill] = (this.tempScores[skill] || 0) + value;
            }

            // Show intermediate feedback if needed, but for now just proceed
            this.currentStepId = option.result;
            this.updateStep();
        },

        finishCase(success, msg) {
            this.feedbackMsg = msg || "Simulación terminada.";
            this.feedbackType = success ? 'good' : 'neutral';

            // Update permanent stats
            if (success) {
                for (const [skill, value] of Object.entries(this.tempScores)) {
                    this.userStats[skill] += value;
                }
                this.currentCase.completed = true;
            }

            this.showFeedbackModal = true;
        },

        // Computed properties (simulated as methods for Alpine x-text/x-bind)
        getProgress() {
            const total = this.availableCases.length || 5;
            const completed = this.availableCases.filter(c => c.completed).length;
            return Math.round((completed / total) * 100);
        },

        getTotalScore() {
            return Object.values(this.userStats).reduce((a, b) => a + b, 0);
        },

        closeFeedback() {
            this.showFeedbackModal = false;

            // Check if all cases completed
            if (this.availableCases.every(c => c.completed)) {
                this.sessionCompleted = true;
                this.view = 'results';
            } else {
                this.view = 'cases-list';
            }
        },

        getScoreColor(score) {
            if (score > 20) return 'text-green-600';
            if (score < 0) return 'text-red-500';
            return 'text-yellow-600';
        },

        async generatePDF() {
            const { jsPDF } = window.jspdf;
            const doc = new jsPDF();

            // Header
            doc.setFillColor(57, 169, 0); // SENA Green
            doc.rect(0, 0, 210, 20, 'F');
            doc.setTextColor(255, 255, 255);
            doc.setFontSize(16);
            doc.text("Reporte de Competencias - RolPlay EDU", 105, 13, { align: "center" });

            // User Info
            doc.setTextColor(0, 0, 0);
            doc.setFontSize(12);
            doc.text(`Nombre del Aprendiz: ${this.playerName}`, 20, 40);
            doc.text(`Sector Productivo: ${this.selectedSector}`, 20, 50);
            doc.text(`Fecha: ${new Date().toLocaleDateString()}`, 20, 60);

            // Stats
            doc.setFontSize(14);
            doc.text("Resultados por Competencia", 20, 80);

            let y = 90;
            for (const [skill, score] of Object.entries(this.userStats)) {
                doc.setFontSize(12);
                doc.text(`${skill}:`, 20, y);

                // Draw bar
                const barWidth = Math.max(0, Math.min(100, score + 50)); // Normalize somewhat
                doc.setFillColor(score >= 0 ? 57 : 220, score >= 0 ? 169 : 50, score >= 0 ? 0 : 50);
                doc.rect(70, y - 5, barWidth, 6, 'F');
                doc.text(`${score}`, 180, y);
                y += 10;
            }

            // Recommendations
            doc.setFontSize(14);
            doc.text("Análisis de Desempeño", 20, y + 20);

            const entries = Object.entries(this.userStats);
            const best = entries.reduce((a, b) => a[1] > b[1] ? a : b);
            const worst = entries.reduce((a, b) => a[1] < b[1] ? a : b);

            doc.setFontSize(11);
            const splitTextBest = doc.splitTextToSize(`Fortaleza: Demostraste un gran desempeño en ${best[0]}, lo que indica que tienes facilidad para manejar situaciones relacionadas con esta competencia en un entorno profesional.`, 170);
            doc.text(splitTextBest, 20, y + 30);

            const splitTextWorst = doc.splitTextToSize(`Oportunidad de Mejora: En ${worst[0]}, los resultados sugieren que podrías beneficiarte de más práctica. Busca situaciones donde puedas ejercitar esta habilidad.`, 170);
            doc.text(splitTextWorst, 20, y + 50);

            // Footer
            doc.setFontSize(10);
            doc.setTextColor(100);
            doc.text("Generado por RolPlay EDU - Producto Técnico Pedagógico SENA", 105, 280, { align: "center" });

            doc.save(`Reporte_RolPlay_${this.playerName}.pdf`);
        }
    }));
});
